<?php
/*
Plugin Name: DigitalOnion Step Blocks
Description: 手順やプロセスを視覚的に分かりやすく表示するためのステップブロックプラグイン。1行あたりの表示数を自由に設定可能です。
Version: 1.0
Author: AwajiTamao
*/

if (!defined('ABSPATH')) exit;

class DigitalOnionStepBlocks {

    public function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_shortcode('digitalonion_step', [$this, 'render_shortcode']);
    }

    public function add_admin_menu() {
        add_menu_page('Step Blocks 設定', 'Step Blocks', 'manage_options', 'digital-onion-step-settings', [$this, 'settings_page'], 'dashicons-editor-ol');
    }

    public function enqueue_admin_assets($hook) {
        if ($hook != 'toplevel_page_digital-onion-step-settings') return;
        wp_enqueue_media();
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
    }

    public function settings_page() {
        if (isset($_POST['save_digital_onion_steps'])) {
            update_option('digital_onion_step_data', $_POST['step_data']);
            echo '<div class="updated"><p>設定を保存しました。</p></div>';
        }
        $data = get_option('digital_onion_step_data', []);
        ?>
        <div class="wrap">
            <h1>DigitalOnion Step Blocks 設定</h1>
            <form method="post" action="">
                <?php for ($i = 1; $i <= 5; $i++): ?>
                    <div style="border: 1px solid #ccc; padding: 15px; margin-bottom: 20px; background: #fff; border-radius: 5px;">
                        <h2>パターン <?php echo $i; ?> (<code>[digitalonion_step id="<?php echo $i; ?>"]</code>)</h2>
                        
                        <label>背景色: </label>
                        <input type="text" name="step_data[<?php echo $i; ?>][bg_color]" class="color-picker" value="<?php echo esc_attr($data[$i]['bg_color'] ?? '#f2f2f2'); ?>">
                        
                        &nbsp;&nbsp;
                        <label>番号形式: </label>
                        <select name="step_data[<?php echo $i; ?>][format]">
                            <option value="STEP" <?php selected($data[$i]['format'] ?? '', 'STEP'); ?>>STEP①</option>
                            <option value="NUM" <?php selected($data[$i]['format'] ?? '', 'NUM'); ?>>①</option>
                        </select>

                        &nbsp;&nbsp;
                        <label>1行の表示数 (PC): </label>
                        <input type="number" name="step_data[<?php echo $i; ?>][columns]" value="<?php echo esc_attr($data[$i]['columns'] ?? '3'); ?>" min="1" max="10" style="width: 50px;">
                        
                        <hr>
                        <div id="steps-container-<?php echo $i; ?>">
                            <?php 
                            $steps = $data[$i]['items'] ?? [['title'=>'','img'=>'','desc'=>'']];
                            foreach ($steps as $idx => $step): ?>
                                <div class="step-item" style="border-bottom: 1px dashed #ddd; padding: 10px 0;">
                                    <strong>ステップ <?php echo $idx + 1; ?></strong><br>
                                    タイトル: <input type="text" name="step_data[<?php echo $i; ?>][items][<?php echo $idx; ?>][title]" value="<?php echo esc_attr($step['title'] ?? ''); ?>" style="width:70%">
                                    <div style="margin: 10px 0;">
                                        <input type="hidden" name="step_data[<?php echo $i; ?>][items][<?php echo $idx; ?>][img]" value="<?php echo esc_attr($step['img'] ?? ''); ?>" class="img-id">
                                        <button class="upload-img button">画像を選択</button>
                                        <div class="img-preview" style="display:inline-block; vertical-align:middle; margin-left:10px;">
                                            <?php if(!empty($step['img'])) echo wp_get_attachment_image($step['img'], [50, 50]); ?>
                                        </div>
                                    </div>
                                    内容: <textarea name="step_data[<?php echo $i; ?>][items][<?php echo $idx; ?>][desc]" style="width:100%; height:60px;"><?php echo esc_textarea($step['desc'] ?? ''); ?></textarea>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <button type="button" class="add-step button" data-pattern="<?php echo $i; ?>" style="margin-top:10px;">+ 段階を追加</button>
                    </div>
                <?php endfor; ?>
                <input type="submit" name="save_digital_onion_steps" class="button-primary" value="すべての設定を保存">
            </form>
        </div>
        <script>
        jQuery(document).ready(function($){
            $('.color-picker').wpColorPicker();
            $(document).on('click', '.upload-img', function(e) {
                e.preventDefault();
                let btn = $(this);
                let frame = wp.media({ title: '画像を選択', button: { text: '使用する' }, multiple: false });
                frame.on('select', function() {
                    let attachment = frame.state().get('selection').first().toJSON();
                    btn.siblings('.img-id').val(attachment.id);
                    btn.siblings('.img-preview').html('<img src="'+attachment.sizes.thumbnail.url+'" width="50" height="50" style="object-fit:cover;" />');
                }).open();
            });
            $('.add-step').click(function(){
                let p = $(this).data('pattern');
                let count = $('#steps-container-'+p+' .step-item').length;
                let html = `<div class="step-item" style="border-bottom: 1px dashed #ddd; padding: 10px 0;">
                    <strong>ステップ ${count + 1}</strong><br>
                    タイトル: <input type="text" name="step_data[${p}][items][${count}][title]" value="" style="width:70%"><br>
                    <div style="margin: 10px 0;">
                        <input type="hidden" name="step_data[${p}][items][${count}][img]" value="" class="img-id">
                        <button class="upload-img button">画像を選択</button>
                        <div class="img-preview" style="display:inline-block; vertical-align:middle; margin-left:10px;"></div>
                    </div>
                    内容: <textarea name="step_data[${p}][items][${count}][desc]" style="width:100%; height:60px;"></textarea>
                </div>`;
                $('#steps-container-'+p).append(html);
            });
        });
        </script>
        <?php
    }

    public function render_shortcode($atts) {
        $atts = shortcode_atts(['id' => '1'], $atts);
        $data = get_option('digital_onion_step_data', []);
        $pattern = $data[$atts['id']] ?? null;
        if (!$pattern || empty($pattern['items'])) return '';

        $bg_color = $pattern['bg_color'] ?? '#f2f2f2';
        $format = $pattern['format'] ?? 'STEP';
        $columns = intval($pattern['columns'] ?? 3);
        
        ob_start();
        ?>
        <style>
            .digitalonion-step-wrapper { 
                display: grid; 
                /* カラム幅を均等にするため 1fr を使用 */
                grid-template-columns: repeat(<?php echo $columns; ?>, 1fr); 
                gap: 24px; 
                width: 100%; 
                max-width: 1200px;
                margin: 3em auto; 
                box-sizing: border-box;
            }
            .digitalonion-step-card { 
                background: <?php echo $bg_color; ?>; 
                padding: 24px; 
                border-radius: 8px; 
                display: flex; 
                flex-direction: column; 
                position: relative;
                box-shadow: 0 2px 8px rgba(0,0,0,0.05);
                /* カード自体の幅が崩れないように固定 */
                width: 100%;
                box-sizing: border-box;
            }
            .digitalonion-step-num { 
                font-size: 1.5rem; 
                font-weight: bold; 
                margin-bottom: 8px; 
                color: #333; 
                opacity: 0.6;
            }
            .digitalonion-step-card h3 { 
                margin: 0 0 16px 0; 
                font-size: 1.15rem; 
                font-weight: bold;
                border-bottom: 1px solid rgba(0,0,0,0.1); 
                padding-bottom: 8px; 
                line-height: 1.4;
            }
            .digitalonion-step-img-container { 
                width: 100%; 
                aspect-ratio: 3/2; 
                overflow: hidden; 
                border-radius: 4px; 
                margin-bottom: 16px; 
            }
            .digitalonion-step-img { width: 100%; height: 100%; object-fit: cover; }
            
            .digitalonion-step-desc { 
                font-size: 0.95rem; 
                line-height: 1.8; 
                color: #444; 
                word-break: break-all;
            }
            
            @media (max-width: 959px) {
                .digitalonion-step-wrapper { 
                    grid-template-columns: repeat(2, 1fr);
                    width: 94%;
                    gap: 16px;
                }
            }

            @media (max-width: 599px) {
                .digitalonion-step-wrapper { 
                    display: flex;
                    flex-direction: column; 
                    gap: 40px; 
                }
                .digitalonion-step-card { 
                    flex-direction: row; 
                    flex-wrap: wrap; 
                    padding: 20px 15px; 
                }
                .digitalonion-step-header { 
                    position: absolute; 
                    top: -30px; 
                    left: 0; 
                    font-size: 1rem; 
                    font-weight: bold; 
                    color: #333; 
                }
                .digitalonion-step-card-left { width: 35%; }
                .digitalonion-step-card-right { width: 60%; padding-left: 5%; display: flex; align-items: center; }
                .digitalonion-step-img-container { aspect-ratio: 2/3; margin-bottom: 0; }
                .digitalonion-step-num, .digitalonion-step-card h3 { display: none; }
            }
            @media (min-width: 600px) { .is-mobile-only { display: none; } }
        </style>
        <div class="digitalonion-step-wrapper">
            <?php foreach ($pattern['items'] as $index => $item): 
                $num = $index + 1;
                $num_circle = $this->get_circled_num($num);
                $label = ($format === 'STEP') ? "STEP{$num_circle}" : "{$num_circle}";
            ?>
                <div class="digitalonion-step-card">
                    <div class="digitalonion-step-header is-mobile-only"><?php echo $label; ?> <?php echo esc_html($item['title']); ?></div>
                    
                    <div class="digitalonion-step-card-left">
                        <div class="digitalonion-step-num"><?php echo $label; ?></div>
                        <h3><?php echo esc_html($item['title']); ?></h3>
                        <div class="digitalonion-step-img-container">
                            <?php if(!empty($item['img'])): ?>
                                <?php echo wp_get_attachment_image($item['img'], 'large', false, ['class' => 'digitalonion-step-img']); ?>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="digitalonion-step-card-right">
                        <div class="digitalonion-step-desc"><?php echo nl2br(esc_html($item['desc'])); ?></div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    private function get_circled_num($n) {
        $map = [1=>'①', 2=>'②', 3=>'③', 4=>'④', 5=>'⑤', 6=>'⑥', 7=>'⑦', 8=>'⑧', 9=>'⑨', 10=>'⑩'];
        return $map[$n] ?? $n;
    }
}
new DigitalOnionStepBlocks();